extensions [ array ] ; Allows us to use arrays to store information during the simulation

__includes [ "F.nls" ]

globals [           ; Variables available to all objects
  total-habitats    ; patches are assigned a unique ID
  next-whatzit-ID   ; whatzits are assigned a unique ID
  step              ; the time
  percent-good      ; percent of patches that are good
  percent-bad       ; percent of patches that are bad
  die-dispersing    ; holds the number of dispersers that die dispersing each year
  Female            ; value for females
  Male              ; value for males
  travel-success    ; keep track of how many whatzits successfully traveled from one patch to another
  migration-success ; keep track of how many whatzits successfully migrated (traveled and remained at least a year)
  geneflow-success  ; keep track of how many whatzits successfully migrated and reproduced
  total-good
  total-OK
  total-bad
  potential-homes   ; identifies available patches 
  Born_this_step    ; count of new births
  Repro-Fem         ; count of reproductive females
]

patches-own [
  tmp1             ; a general-purpose temporary variable
  tmp2             ; a general-purpose temporary variable
  quality          ; -1 poor, 0 OK, 1 great
  tagged           ; used in the process of finding contiguous areas
  habitat-ID       ; pointer to the habitat that groups this patch with others
  number-leaving   ; to hold the total number of whatzits leaving - including those that die
  number-arriving  ; to hold the total number of whatzits arriving
  territory-value  ; the average value of patches within territory-radius
  male-owner       ; which male, if any, claims this patch?
  female-owner     ; which female, if any, claims this patch?
  #-breeding-males ; how many breeding males are available to a female on this patch?
  #-breeding-fem   ; how many breeding females are available to a male on this patch?
  male-terr-potential   ; could a male whatzit move here w/out creating territory overlap?
  female-terr-potential ; could a female whatzit move here w/out creating territory overlap?
]

breed [ whatzits whatzit ]

whatzits-own [
  ID
  age
  sex
  pregnant?               ; boolean 
  have-offspring?         ; boolean
  pregnant-from           ; what is the current father's lineage ( can only be one )
  father-ID               ; the father whatzit
  mate-attr               ; how attractive a mate I am
  lineage                 ; my ancestry
  survival-index          ; used to calculate death
  unsettled?              ; boolean to indicate that I need to search out a better patch
  dispersed?              ; boolean to indicate that I did search
  can-stay?               ; boolean to indicate that can live on the current patch
  last-habitat-home       ; last habitat that I remained in for at least a year
  current-habitat-home    ; the habitat that I've been occupying most recently
  birth-habitat           ; habitat where I was born
  years-here              ; years that I have been in current habitat
  F                       ; Wright's inbreeding coefficient
  total-steps             ; Number of migration steps
  mates-birthplace        ; ID of habitat where the male mate was born
  starting-patch          ; Used to track where the whatzit started each step
  gene-markers            ; Keep track of alleles on chromosomes
]

breed [ habitats habitat ]

habitats-own [
  ID                ; my habitat ID
  extent            ; how many contiguous patches are associated
  link-patches      ; list of patches that are stepping stones to other habitats
]

breed [ reporters reporter ]

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; INITIALIZATION PROCEDURES
;;     habitats are initialized separately from whatzits
;;     this allows multiple experiments using the same patch configuration
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; The World is blanketed with poor, OK, and good habitats as follows:
;;   1 - All patches are given an unset value (10)
;;   2 - A set of seed patches with random values (-1, 0, and 1) are established with a 
;;       distribution approaching the percentages for good and poor given in the interface
;;   3 - These locations are grown in a random fashion until all patches are set
;;   4 - Each patch is then given a value equal to the number of contiguous patches 
;;       Here, contiguous means sharing an edge (each patch has 4 potential neighbors)

to initialize-patches
  ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  __clear-all-and-reset-ticks
  set Female 2
  set Male 1

  ifelse patch-seed != 0 
  [ type "setting patch seed " print patch-seed random-seed patch-seed ]
  [ random-seed new-seed ]
  
  ; Set patch variables to known value
  ask patches [ set quality 10 set tagged FALSE]
  
  let x seed-patches   
  ; Establish the seed patches
  while [ x > 0 ] [
    ask patch random-pxcor random-pycor [
      let r random 100
      ifelse ( r < seed-good )
        [ set quality 1 ]
        [
          ifelse r < (seed-good + seed-bad )
          [ set quality -1 ]
          [ set quality 0 ]
        ]
      set-color
      set x x - 1
    ]
  ]
   
  ; Grow the seed patches to fill the entire World
  while [ 1 < ( count patches with [ quality = 10 ] ) ]
  [
    ask patches with [ quality = 10 ] [ initialize-patch ]
  ]
  ask patches with [ quality = 10 ] [ initialize-patch ]
   
  ; Identify how many contiguous cells there are in each cell's contiguous habitat
  set-extents
  
  ; Initialize habitat objects to keep track of migration success (and prevent cross-habitat breeding)
  initialize-habitats
  
  ask patches with [ quality = 1 ]
  [
    set territory-value mean [ quality ] of ( patches in-radius territory-radius )
  ]
  
  set percent-good ( 100 * count patches with [quality = 1] / count patches )
  set percent-bad  ( 100 * count patches with [quality = -1] / count patches )
end

to initialize-patch
   let my-neighbor one-of neighbors4
   set quality [quality] of my-neighbor
   set-color
end

to set-color
  ifelse ( -1 = quality )
  [ set pcolor 18 ]
  [
    ifelse ( 0 = quality )
    [ set pcolor 48 ]
    [ 
      if ( 1 = quality)
      [ set pcolor 68 ]
    ]
  ]
end

to initialize-habitats
  ; Initialize arrays for travel, migration, and geneflow success ID
  set travel-success    array:from-list n-values ( total-habitats * total-habitats ) [ 0 ]
  set migration-success array:from-list n-values ( total-habitats * total-habitats ) [ 0 ]
  set geneflow-success  array:from-list n-values ( total-habitats * total-habitats ) [ 0 ]
  
  print "Finding take-off patches for shortest distances to habitats"                   ; This code calculates the shortest distance between all habitat pairs
  let print-count 0
  let from-ID 1
  while [ from-ID <= total-habitats ]                                                   ; Loop through all the habitats (outside loop)
  [                                                                                     ;
    write from-ID type ","                                                              ;    This takes a long, tome, so give the user some feedback
    ifelse ( print-count > 15 )                                                         ;
      [ set print-count 0 print "" ]                                                    ;
      [ set print-count print-count + 1 ]                                               ;
    let from-habitat one-of habitats with [ ID = from-ID ]                              ;    Identify the single "from" habitat from the list 
    ask from-habitat [ set link-patches ( list 0 ) ]                                    ;    Initialize the link-patches list
    let from-patches patches with [ habitat-ID = from-habitat ]                         ;    Get list of all patches associated with the "from" habitat
    let to-ID 1                                                                         ;
    while [ to-ID <= total-habitats ]                                                   ;    Loop through all the habitats  (inside loop )
    [                                                                                   ;
      ifelse (to-ID = from-ID)                                                          ;    If the two habitats are the same, the answer is 0
      [                                                                                 ;
        ask from-habitat [ set link-patches lput 0 link-patches ]                       ;
      ]                                                                                 ;
      [                                                                                 ;
        let to-habitat one-of habitats with [ ID = to-ID ]                              ;      Identify the single "to" habitat from the list
        let to-patches patches with [ habitat-ID = to-habitat ]                         ;      Get a list of all patches associated with the "to" habitat
        ask from-patches                                                                ;      Ask all of the "from" patches to:
        [                                                                               ;         ask all "to" habitats to calculate their dist to me and ID the closest
          set tmp1 min-one-of to-patches [ distance myself ]                            ;         store the distance to that patch in tmp2
          set tmp2 distance tmp1                                                        ;
        ]                                                                               ;
        let closest min-one-of from-patches [ tmp2 ]                                    ;    Find the "from" patch that is closest to any of the "to" patches
        ask from-habitat [ set link-patches lput ( closest ) link-patches ]             ;    Store that patch
      ]                                                                                 ;
      set to-ID to-ID + 1                                                               ;
    ]                                                                                   ;
    ask from-habitat [ set link-patches butfirst link-patches ]                         ;
    set from-ID from-ID + 1                                                             ;
  ]
  
  print ""
  ask habitats [ set hidden? TRUE ]    
end
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; These three procedures clump like-cells together into numbered habitats                                   ;;
;;                                                                                                           ;;
to set-extents                                                                                               ;;
  set total-habitats 0                                                                                       ;;
  let y min-pycor                                                                                            ;;
  while [ y <= max-pycor ]                                                                                   ;;
  [                                                                                                          ;;
    let x min-pxcor                                                                                          ;;
    while [ x <= max-pxcor ]                                                                                 ;;
    [                                                                                                        ;;
      ask patch x y [ find-extent ]                                                                          ;;
      set x x + 1                                                                                            ;;
    ]                                                                                                        ;;
    set y y + 1                                                                                              ;;
  ]                                                                                                          ;;
  ask patches [ if habitat-id = 0 [ set habitat-ID NOBODY ]]                                                 ;;                                                                                                         ;;
end                                                                                                          ;;
                                                                                                             ;;
to find-extent                                                                                               ;;
  if ( habitat-ID != 0 OR quality != 1 ) [ stop ]                                                            ;;
  set total-habitats total-habitats + 1                                                                      ;;
  sprout-habitats 1 [ set ID total-habitats ]                                                                ;;
  let the-habitat one-of habitats with [ ID = total-habitats ]                                               ;;
  tag-neighbors SELF quality                                                                                 ;;
  let the-count count patches with [ tagged = TRUE ]                                                         ;;
  ask the-habitat [ set extent the-count ]                                                                   ;;
  ask the-habitat [ set plabel ID set plabel-color black ]                                                   ;;
  ask patches with [ tagged = TRUE ] [ set habitat-ID the-habitat set tagged FALSE ]                         ;;
end                                                                                                          ;;
                                                                                                             ;;
to tag-neighbors [ start-patch qual ]                                                                        ;;
  let tag-list (list start-patch)                                                                            ;;
  let at-patch 0                                                                                             ;;
  ask start-patch [ set tagged TRUE ]                                                                        ;;
  while [ ( length tag-list ) > 0 ]                                                                          ;;
  [                                                                                                          ;;
    let cur-patch first tag-list                                                                             ;;
    set tag-list butfirst tag-list                                                                           ;;
    ask cur-patch                                                                                            ;;
    [                                                                                                        ;;
       set tag-list lput ( neighbors4 with [ habitat-ID = 0 AND tagged = FALSE AND quality = qual ] ) tag-list
       ask neighbors4 with [ habitat-ID = 0 AND tagged = FALSE AND quality = qual ] [ set tagged TRUE ]      ;;
    ]                                                                                                        ;;
  ]                                                                                                          ;;
end                                                                                                          ;;
;;                                                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Whatzits are placed randomly in an area that is unoccupied within a user-set radius.

to initialize-whatzits
  ifelse NOT ( whatzit-seed = 0 )
  [ type "setting whatzit seed " print whatzit-seed random-seed whatzit-seed ]
  [ random-seed new-seed ]

print "init'ing genetics ... "
  initialize-genetics                 ; initialize a table in the genetics package (F.nls)
  ; Place the initial whatzits
  ask whatzits [ die ]                           ; remove all whatzits
  cd                                             ; remove any drawing
  set step 0
  set next-whatzit-id 1

print "init'ing whatzits ... "
  create-whatzits initial-whatzits
  [ ifelse random 2 = 0 [ set sex  Female  set color 134 ] [ set sex Male set color 94 ]  ; choose random sex
    set years-here 0
    set ID next-whatzit-id
    set lineage next-whatzit-id
    set next-whatzit-id next-whatzit-id + 1
    set age random 10                                                                      ; choose random age
    setxy random-xcor random-ycor
    set pregnant-from 0
    set father-ID 0
    set pregnant? FALSE
    set have-offspring? FALSE
    set unsettled? TRUE
    set gene-markers list ( 1 + 10 * ID ) ( 2 + 10 * ID )  ; e.g. individual 9 gets allele markers 91 and 92
  ]
  ask whatzits
  [
    find-initial-home
    move-to one-of potential-homes

    set birth-habitat habitat-id
    set last-habitat-home 0
    set current-habitat-home habitat-id
    set F 0
  ]
  ask whatzits  [ draw-territory 10 ]

print "setting plots ... "  
  my-setup-plots                  ; The population plots are initialized
print "creating lineage report ... "  
  setup-lineage-report         ; initialize the lineage report file for dead whatzits
end
  
;to find-initial-home
;  without-interruption 
;  [
;    let potential-homes patches with [ quality = 1 AND ( count whatzits in-radius ( territory-radius * 2) = 0 )
;;    let OK-site FALSE
;;    while [ OK-site = FALSE ]
;    [ set potential-homes patches with [ quality = 1 AND ( count whatzits in-radius ( territory-radius * 2) = 0 ) ]
;      ifelse NOT ( allow-M/F-terr-overlap )
;      [
;;      [ ifelse ( quality = 1 AND ( [extent] of [habitat-ID] of patch-here ) >= min-habitat-size AND ( count whatzits in-radius ( territory-radius * 2) = 1 ) )
;;      [ ifelse ( quality = 1 AND ( count whatzits in-radius ( territory-radius * 2) = 1 ) )
;; need to add to above AND "there's enough viable habitat here for me to live on"      [ ifelse ( quality = 1 AND ( [extent] of [habitat-ID] of patch-here ) >= min-habitat-size AND ( count whatzits in-radius ( territory-radius * 2) = 1 ) )
;;        [ set OK-site TRUE ]
;;        [ setxy random-xcor random-ycor ]
;        [ move-to one-of potential-homes ]
;      ]
;      [ let mysex sex
;        ifelse ( quality = 1 AND ( [extent] of [habitat-ID] of patch-here ) >= min-habitat-size AND ( count whatzits in-radius ( territory-radius * 2) with [ sex = mysex ] = 1 ) ) 
;        [ set OK-site TRUE ]
;;        [ setxy random-xcor random-ycor ]
;        [ move-to one-of potential-homes ]
;      ]
;    ]
;  ]
;end

to find-initial-home
  ifelse NOT ( allow-M/F-terr-overlap )
  [ set potential-homes patches with [ quality = 1 AND ( count whatzits in-radius ( territory-radius * 2) = 0 ) ]
  ]
  [ let mysex sex
    set potential-homes patches with [ quality = 1 AND ( count whatzits in-radius ( territory-radius * 2 ) with [ sex = mysex ] = 0 ) ]
  ]
; need to add to above AND "there's enough viable habitat here for me to live on"      [ ifelse ( quality = 1 AND ( [extent] of [habitat-ID] of patch-here ) >= min-habitat-size AND ( count whatzits in-radius ( territory-radius * 2) = 1 ) )

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; WHATZIT GO PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go
  set step step + 1
  tick
;  if is-string? Elim_These_Grn [ alter-habitat ]
  track-movement
  ask patches [ set number-leaving 0 ]
  set-mate-attr
  mate-and-give-birth          
  disperse                                            ; yearling females and all males that were unsuccessful in mating seek new homes
;  optimize-territory
  survive                                             ; age and apply mortality
  clear-drawing
  ask whatzits with [ age > 1 ] [ draw-territory 10 ]
  do-plots
;  save-whatzit-report
end

to track-movement
  ask whatzits                  ; check to see if I've successfully migrated ( remained in a new patch for two years )
  [ ifelse ( current-habitat-home = habitat-ID )
    [ set years-here years-here + 1
    ]
    [ set years-here 0
      set current-habitat-home habitat-ID
    ]
    if ( years-here = 2 )
    [ if NOT ( last-habitat-home = 0 )
      [ increment-migration-success last-habitat-home current-habitat-home  ;; save the info that I've successfully migrated from the last location
      ]
      set last-habitat-home current-habitat-home
    ]
  ]
end

to set-mate-attr
  ask whatzits
  [ set pregnant? FALSE   ; also used in males as a flag that they have or have-not      
    set mate-attr 0
    if ( age >= first-repro-age )  [ set mate-attr 1 ]
    if ( age >= superior-repro-age)  [ set mate-attr 2 ]
    if ( age > last-repro-age)  [ set mate-attr 0 ]
  ]
end

to mate-and-give-birth
  ask whatzits with [ sex = Female AND age >= first-repro-age ] 
  [ let my-habitat current-habitat-home
    let potential-mates ( whatzits in-radius ( mating-radius )) with [ current-habitat-home = my-habitat AND sex = Male AND age >= first-repro-age AND age <= last-repro-age ]       ; get list of local males
    let list-potential-mates sort potential-mates
    if ( count potential-mates > 0 )
    [ let male-attr-sum sum [ mate-attr ] of potential-mates
;      let num-babies-needed round ( birth-rate  * ( 1 - ( F-influence-on-birth * F ) ) )
      let num-babies-needed round ( birth-rate  * ( 1 - ( F * Avg_Lethal_Equivalents ) ) ) ; this is the # of individuals per breeding female, that survive the 1st year
      while [ num-babies-needed > 0 ]
      [ ; Choose a father
        let rand-num random-float male-attr-sum
        let potential-fathers count potential-mates
        let at-father 0
        let cumulative-attr 0
        let success FALSE
        while [ NOT success AND ( at-father < potential-fathers ) ]
        [ set cumulative-attr cumulative-attr + [ mate-attr ] of item at-father list-potential-mates
; type "cum-attr: " print cumulative-attr
; type "at-father: " print at-father
; type "pot-father: " print potential-fathers
; print ""
          if cumulative-attr > rand-num
          [ set success TRUE
            let father item at-father list-potential-mates
            set father-ID father
            set mates-birthplace [birth-habitat] of father         ; where the new father-to-be was birthed
            set have-offspring? TRUE                                                   ; remember that I've had children
            ask father 
            [ set pregnant? TRUE                    ; male remembers he has mated
              set have-offspring? TRUE              ; male remembers that he's had offspring
            ]
            let child-lineage list lineage ( [lineage] of father )
            hatch 1    ; set whatzit variables
            [ ifelse random 2 = 0 [ set sex Female  set color 134 ] [ set sex Male set color 94 ]
              set age 0
              set ID next-whatzit-id  set next-whatzit-id next-whatzit-id + 1
              set lineage trimmed-lineage ( fput ID child-lineage ) lineage-depth 0
              set birth-habitat habitat-id
              set last-habitat-home 0
              set current-habitat-home habitat-id
              set years-here 0
              set pregnant? FALSE
              set have-offspring? FALSE
              set unsettled? FALSE
              set pregnant-from 0
              set father-ID 0
              set F Wrights-inbreeding lineage lineage-depth                    ; calculate inbreeding coefficient
              set gene-markers select-parent-genes lineage
            ]
            increment-geneflow-success ([ID] of birth-habitat) ([ID] of current-habitat-home)
            increment-geneflow-success ( [ID] of mates-birthplace ) ([ID] of current-habitat-home)
          ]
          set at-father at-father + 1
        ]
        set num-babies-needed num-babies-needed - 1
      ]
    ]
  ]
end

to disperse
  ask whatzits                                             ; start by assuming that each does not disperse/relocate
  [ set dispersed? FALSE
    set unsettled? FALSE
    set total-steps 0
    set starting-patch patch-here
  ] 
  set die-dispersing 0                                     ; initialize the number that die dispersing to 0

  ; identify dispersing males
  If males-disperse = "1st_repro_age_only" [ ask whatzits with [ ( sex = Male AND age = first-repro-age ) ]  [ set unsettled? TRUE ] ]
  If males-disperse = "1st_repro_&_if_no_mates_here" [ ask whatzits with [ ( sex = Male AND NOT pregnant? AND age >= first-repro-age OR age = first-repro-age) ]  [ set unsettled? TRUE ] ]

  ; identify dispersing females
  If females-disperse = "1st_repro_age_only" [ ask whatzits with [ ( sex = Female AND age = first-repro-age ) ]  [ set unsettled? TRUE ] ]
  If females-disperse = "1st_repro_&_if_no_mates_here" [ ask whatzits with [ ( sex = Female AND NOT pregnant? AND age >= first-repro-age OR age = first-repro-age) ]  [ set unsettled? TRUE ] ]

;  set potential-homes patches with [ quality = 1 AND ( count whatzits in-radius ( territory-radius * 2) = 0 ) ]
;  if count whatzits with [ unsettled? = TRUE ] > count potential-homes
;  [ print "not enough available territories; dispersal aborted" 
;    stop
;  ]

  while [ count whatzits with [ unsettled? ] > 0 ]
  [ ask max-one-of whatzits with [ unsettled? ] [ age ]    ; Pick the oldest unsettled whatzit
    [ move
      evaluate-habitat
      evaluate-neighbors
      settle-or-not
    ]
  ]
end
     
to move
  set dispersed? TRUE                                  ; To remember that I dispersed
  If Dispersal = "Random"
  [ lt random 90  rt random 90                           ; point in a random direction with max turn of 90 degrees
    forward  1                                           ; move forward one step
  ]
  If Dispersal = "Other (see code)"
  ; whatzits have 2:1 odds of stepping to "best quality available" adjacent patch.  , and differentiate between red/yellow based on their relative danger.
  [ let odds ( 3 * ( death-rate-yel + death-rate-red ) )
    let rand random odds
    ifelse rand > ( odds - death-rate-red )
    [ face min-one-of neighbors [ quality ] 
    ]
    [ ifelse rand < ( odds - ( death-rate-yel + death-rate-red ) )
      [ face max-one-of neighbors [ quality ] 
      ]
      [ let rem-neighbors neighbors
        set rem-neighbors sort rem-neighbors 
        set rem-neighbors remove-item 7 rem-neighbors 
        set rem-neighbors remove-item 6 rem-neighbors 
        set rem-neighbors remove-item 0 rem-neighbors 
        set rem-neighbors remove-item 0 rem-neighbors 
        face one-of rem-neighbors
      ]
    ]
  forward 1
  ]
end

to evaluate-habitat
  set can-stay? TRUE     
  if NOT ( quality = 1 )
  [ set can-stay? FALSE ]                               ; if the patch is not good quality I'll remain unsettled

  ; check to make sure there's enough good habitat here
  if (can-stay? = TRUE)
  [ let potential-territory-patches patches in-radius ( territory-radius )
    let terr-count count potential-territory-patches with [ quality = 1 ]
    if terr-count < ( 0.8 * ( territory-radius ^ 2 ) * pi ) ; make proportion user-spec'd
    [ set can-stay? FALSE ]                             ; if it doesn't look like there's much good habitat here I'll remain unsettled
  ]
end

to evaluate-neighbors      ; if quality and area here look ok ...
  if (can-stay? = TRUE)
  [ let me self
    let myage age

  ; dominant animals claim territory, subordinate ones move
    ifelse NOT ( allow-M/F-terr-overlap )
    [ let locals whatzits in-radius (territory-radius * 2) with [ self != me ]                         ; list of local whatzits, excluding myself
      if ( ( count locals with [ age >= myage ] ) > 0 ) [ set can-stay? FALSE ]                        ; Can't stay with older whatzits
      ask locals with [ age < myage ]  [ set unsettled? TRUE ]                                         ; Kick out local younger whatzits
    ]
    [ let local-males   whatzits in-radius (territory-radius * 2) with [ sex = Male AND self != me ]    ; list of local males, excluding myself
     let local-females whatzits in-radius (territory-radius * 2) with [ sex = Female AND self != me ]  ; list of local females, excluding myself
     if (sex = Male)
      [ if ( count local-males > 0 )
        [ if age <= [age] of max-one-of local-males [ age ] [ set can-stay? FALSE ]                      ; Can't stay with older males
          ask local-males with [ age < myage ] [ set unsettled? TRUE ]                                   ; Kickout local younger males
        ]
      ]

      if (sex = Female )
      [ if ( count local-females > 0 )
        [ if age <= [age] of max-one-of local-females [ age ] [ set can-stay? FALSE ]                    ; Can't stay with older females
          ask local-females with [ age < myage ] [ set unsettled? TRUE ]                                 ; Kickout local younger females
        ]
      ]
    ]
  ]
end

to settle-or-not
  if (can-stay? = TRUE)
  [ set unsettled? FALSE                               ; I'll stay here, for now
    increment-travel-success starting-patch patch-here ; Keep track of my travels
    set starting-patch patch-here                      ; I might be unsettled again and travel to another patch from here
  ]
     
  if unsettled?                                          ; if I remain unsettled, I adjust steps
  [  let survive-prob random-float 1                     ;    - then I repeat the process
    if ( ( quality = 0 AND survive-prob <= death-rate-grn ) OR ( quality = 1 AND survive-prob <= death-rate-yel ) OR ( quality = 2 AND survive-prob <= death-rate-red ) )
    [ set die-dispersing die-dispersing + 1
      process-death
    ]
  ]
end

to-report trimmed-lineage [ tree max-depth cur-depth ]
  if is-number? tree      [ report tree ]
  if ( cur-depth = max-depth ) [ report item 0 tree ]
  set cur-depth cur-depth + 1
  report ( sentence
    (list item 0 tree
    trimmed-lineage (item 1 tree) max-depth cur-depth
    trimmed-lineage (item 2 tree) max-depth cur-depth ))
end

to-report select-parent-genes [ line ]
  let mom 0
  let dad 0
  ifelse is-number? ( item 1 line ) [ set mom item 1 line ] [ set mom item 0 item 1 line ]
  ifelse is-number? ( item 2 line ) [ set dad item 2 line ] [ set dad item 0 item 2 line ]
  let mom-genes [ gene-markers ] of one-of whatzits with [ ID = mom ]
  let dad-genes [ gene-markers ] of one-of whatzits with [ ID = dad ]
  let my-genes ( list )
  let j length mom-genes
  let i 0
  while [ i < j ]
  [
    set my-genes lput item ( i + random 2 ) mom-genes my-genes
    set my-genes lput item ( i + random 2 ) dad-genes my-genes
    set i i + 2
  ]
  report my-genes
end

to optimize-territory  

  ; for each unoccupied habitable patch, 
  ask patches with [ quality = 1 and whatzits-here = nobody ]
  [ let this-habitat habitat-ID

   ; count how many breeding opportunities it offers,
    set #-breeding-males count ( whatzits in-radius ( mating-radius )) with [ current-habitat-home = this-habitat AND sex = Male AND age >= first-repro-age AND age <= last-repro-age ]
    set #-breeding-fem count ( whatzits in-radius ( mating-radius )) with [ current-habitat-home = this-habitat AND sex = Female AND age >= first-repro-age AND age <= last-repro-age ]

    ; then note who "owns" it (if anyone),
    let owners whatzits in-radius ( territory-radius ) with [ age > 1 ] 
    set male-owner [ who ] of owners with [ sex = male ] 
    set female-owner [ who ] of owners with [ sex = female ] 
 
    ; then decide if a whatzit could move here without creating territory conflicts
    let male-overlaps [ male-owner ] of ( patches in-radius territory-radius )
    set male-overlaps remove nobody male-overlaps
    set male-overlaps remove 0 male-overlaps 
    set male-overlaps sort male-overlaps 
    if first male-overlaps = last male-overlaps [ set male-terr-potential 1 ]    

    let female-overlaps [ female-owner ] of ( patches in-radius territory-radius )
    set female-overlaps remove nobody female-overlaps
    set female-overlaps remove 0 female-overlaps 
    set female-overlaps sort female-overlaps 
    if first female-overlaps = last female-overlaps [ set female-terr-potential 1 ]
  ]

  ask whatzits with [ sex = female and age > 1]
  [ let me ID
    let breeding-males-here [ #-breeding-males ] of patch-here 
    let my-best-patches patches with [ female-owner = me AND #-breeding-males >= breeding-males-here AND female-terr-potential = 1  ]; of those, list only the ones that offer > or = # of mating opportunities and don't overlap competitors' territories
    move-to max-one-of my-best-patches [ territory-value]  ; of those, move to the one with the highest territory-value  
  ]

  ask whatzits with [ sex = male and age > 1]
  [ let me ID
    let breeding-females-here [ #-breeding-fem ] of patch-here 
    let my-best-patches patches with [ male-owner = me AND #-breeding-fem >= breeding-females-here AND male-terr-potential = 1  ]; of those, list only the ones that offer > or = # of mating opportunities and don't overlap competitors' territories
    move-to max-one-of my-best-patches [ territory-value]  ; of those, move to the one with the highest territory-value
  ]
end

to survive
  ask whatzits
  [ set age age + 1
    if (age > 1) 
    [ let territory-quality ( 1 + mean [ quality ] of patches in-radius territory-radius ) / 2
      if ( ( random-float 1.0 ) > ( territory-quality * annual-survival )  ) [ process-death ]
      ; S * e ^ (-F * B)  ;  ( B ranges from 0 to 13, S = annual survival rate, F is 0-.5 )  ; only applies to first year survival
    ]
  ]
end

to process-death
  if ( have-offspring? ) [ report-lineage ]
  die
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Travel, migration, and geneflow success array management
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to increment-travel-success [ start-patch end-patch ]
  let index ( ( ( [ID] of [habitat-ID] of start-patch ) - 1 ) * total-habitats ) + ( [ID] of [habitat-ID] of end-patch - 1)
  array:set travel-success index  ( array:item travel-success index + 1 )
end

to-report get-travel-success [ start-habitat end-habitat  ]
  let index ( ( start-habitat - 1 ) * total-habitats ) + ( end-habitat - 1)
  report array:item travel-success index
end

to increment-migration-success [ start-habitat end-habitat ]
  let index ( ( ([ID] of start-habitat) - 1 ) * total-habitats ) + ( ( [ID] of end-habitat - 1 ) )
  array:set migration-success index  ( array:item migration-success index + 1 )
end

to-report get-migration-success [ start-habitat end-habitat ]
  let index ( ( start-habitat - 1 ) * total-habitats ) + ( end-habitat - 1)
ifelse (array:length migration-success <= index)
[ type index  report 9999 ]
[ report array:item migration-success index
  ]
end

to increment-geneflow-success [ start-habitat end-habitat ]
  let index ( ( start-habitat - 1 ) * total-habitats ) + ( end-habitat - 1 )
  array:set geneflow-success index  ( array:item geneflow-success index + 1 )
end

to-report get-geneflow-success [ start-habitat end-habitat ]
  let index ( ( start-habitat - 1 ) * total-habitats ) + ( end-habitat - 1)
  report array:item geneflow-success index
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; PLOTTING PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to my-setup-plots                             ;; set up plotting
  set-current-plot "Population"
  set-plot-y-range 0 100
  plot-pen-reset
  clear-all-plots
  
  set-current-plot "Average Age"
  set-plot-y-range 0 20
  plot-pen-reset
  clear-all-plots

  set-current-plot "Success in Dispersing"
  set-plot-y-range 0 100
  plot-pen-reset
  clear-all-plots

  set-current-plot "Mated"
  set-plot-y-range 0 100
  plot-pen-reset
  clear-all-plots

  set-current-plot "Inbreeding"
  set-plot-y-range 0 1
  plot-pen-reset
  clear-all-plots
end

to do-plots
  ;;  Plot Population
  set-current-plot "Population"
  set-current-plot-pen "males" plot ( count whatzits with [ sex = Male ] )
  set-current-plot-pen "females" plot ( count whatzits with [ sex = Female  ] )

  ;; Plot mean age
  let males whatzits with [ sex = Male ]
  let females whatzits with [ sex = Female  ] 
  set-current-plot "Average Age"
  if count males > 0
  [ set-current-plot-pen "males" plot ( mean [age] of males )
  ]
  if count females > 0
  [ set-current-plot-pen "females" plot ( mean [age] of females )
  ]

  ;; Plot number dispersed
  set-current-plot "Success in Dispersing"
  set-current-plot-pen "males" plot ( count whatzits with [ dispersed? AND sex = Male ] )
  set-current-plot-pen "females" plot ( count whatzits with [ dispersed? AND sex = Female ] )
  set-current-plot-pen "die dispersing" plot ( die-dispersing )

  ;; Plot mating success
  set-current-plot "Mated"
  set-current-plot-pen "males" plot ( count whatzits with [ pregnant? AND sex = Male ] )
  set-current-plot-pen "females" plot ( count whatzits with [ pregnant? AND sex = Female ] )

  ;; Plot inbreeding coefficient
  set-current-plot "Inbreeding"
  if (count whatzits with [ F >= 0 ] > 0 )
  [ set-current-plot-pen "F" plot ( mean [ F ] of whatzits with [ F >= 0 ] ) ]
  
  set Born_this_step count whatzits with [ age = 1 ]
  set Repro-Fem count whatzits with [ sex = female and age >= first-repro-age and age <= last-repro-age]
end

to draw-territory [ points ]  ; for visualization, whatzits draw a territory circle around themselves
  let angle 360 / points
  let dist 2 * sin ( angle / 2 ) * territory-radius
  fd territory-radius
  rt 90 - ( angle / 2 )
  pen-down
  let point points
  while [ point > 0 ]
  [ rt angle
    fd dist
    set point point - 1
  ]
  rt angle / 2 + 90
  pen-up 
  fd territory-radius
  rt 180
end
    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; REPORT PROCEDURES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to generate-report
  if choose-report = "Lineage"                         [ save-lineage-report ]
  if choose-report = "DL Travel (binary)"              [ save-DL-travel TRUE ]
  if choose-report = "DL Travel (value)"               [ save-DL-travel FALSE ]
  if choose-report = "DL Migration (binary)"           [ save-DL-migration TRUE ]
  if choose-report = "DL Migration (value)"            [ save-DL-migration FALSE ]
  if choose-report = "DL Gene Flow (binary)"           [ save-DL-geneflow TRUE ]
  if choose-report = "DL Gene Flow (value)"            [ save-DL-geneflow FALSE ]
  if choose-report = "DL distance"                     [ save-DL-distance ]
  if choose-report = "DL % bad habitat between sites"  [ save-DL-pct "bad" ]
  if choose-report = "DL % good habitat between sites" [ save-DL-pct "good" ]
  if choose-report = "All Reports" 
  [ save-lineage-report
    save-DL-travel TRUE
    save-DL-travel FALSE
    save-DL-migration TRUE
    save-DL-migration FALSE
    save-DL-geneflow TRUE
    save-DL-geneflow FALSE
    save-DL-distance
    save-DL-pct "bad"
    save-DL-pct "good"
  ]
end

to setup-lineage-report
  print "Lineage report on dead individuals will be sent to: 'report-lineage-1.txt'"
  let file-name "report-lineage-1.txt"
  if file-exists? file-name [ file-delete file-name ]
  file-open file-name
  file-type "ID, Father, Mother, Sex, F, A1, A2"
  file-print ""
  file-close
end

to report-lineage
  file-open "report-lineage-1.txt"
  let father 0
  let mother 0
  let me 0
  ifelse (is-list? lineage )
  [ set me item 0 lineage
    ifelse is-number? ( item 2 lineage ) [ set father item 2 lineage ] [ set father item 0 ( item 2 lineage ) ]
    ifelse is-number? ( item 1 lineage ) [ set mother item 1 lineage ] [ set mother item 0 ( item 1 lineage ) ]
  ]
  [ set me lineage ]
  file-type (word me "," father "," mother "," sex "," ( precision F 10 ))
  foreach gene-markers [ file-type "," file-type ? ]
  file-print ""
  file-close
end

to save-lineage-report
  file-close
  print "Lineage report on dead individuals saved to: 'report-lineage-1.txt'"
  print "Lineage report on live individuals saved to: 'report-lineage-2.txt'"
  let file-name "report-lineage-2.txt"
  if file-exists? file-name [ file-delete file-name ]
  file-open file-name
  ask whatzits
  [ let father 0
    let mother 0
    let me 0
    ifelse (is-list? lineage )
    [ set me item 0 lineage
      ifelse is-number? ( item 2 lineage ) [ set father item 2 lineage ] [ set father item 0 ( item 2 lineage ) ]
      ifelse is-number? ( item 1 lineage ) [ set mother item 1 lineage ] [ set mother item 0 ( item 1 lineage ) ]
    ]
    [ set me lineage ]
    file-type (word me "," father "," mother "," sex "," ( precision F 10 )) 
    foreach gene-markers [ file-type "," file-type ? ]
    file-print ""
  ]
  file-close
end

to setup-DL [ file-name ]
  file-close
  type "Report saved to: "   print file-name
  if file-exists? file-name [ file-delete file-name ]
  file-open file-name
  file-print "DL"
  file-type "n=" file-print total-habitats
  file-print "format=fullmatrix"
  file-print "  labels:"
  let from-ID 1
  while [ from-ID <= total-habitats ]
  [  file-type "Site" file-type from-ID
    if (from-ID != total-habitats) [ file-type "," ]
    set from-ID from-ID + 1
  ]
  file-print ""
  file-print "  data:"
end

to save-DL-geneflow [ binary ]
  ifelse binary 
    [ setup-DL "DL-geneflow-binary.txt" ]
    [ setup-DL "DL-geneflow-value.txt" ]
  let from-ID 1
  while [ from-ID <= total-habitats ]                                         ; Loop through all the habitats (outside loop)
  [ file-type "  "
    let to-ID 1
    while [ to-ID <= total-habitats ]                                         ;    Loop through all the habitats  (inside loop )
    [ let success get-geneflow-success from-ID to-ID
      ifelse binary 
        [ ifelse success > 0 [ file-type 1 ] [ file-type 0 ] ]
        [ file-type success ]

      file-type " "
      set to-ID to-ID + 1
    ]
    file-print ""
    set from-ID from-ID + 1
  ]
  file-close
end

to save-DL-migration [ binary ]
  ifelse binary 
    [ setup-DL "DL-migration-binary.txt" ]
    [ setup-DL "DL-migration-value.txt" ]
  let from-ID 1
  while [ from-ID <= total-habitats ]                                         ; Loop through all the habitats (outside loop)
  [ file-type "  "
    let to-ID 1
    while [ to-ID <= total-habitats ]                                         ;    Loop through all the habitats  (inside loop )
    [ let success get-migration-success from-ID to-ID
      ifelse binary 
        [ ifelse success > 0 [ file-type 1 ] [ file-type 0 ] ]
        [ file-type success ]

      file-type " "
      set to-ID to-ID + 1
    ]
    file-print ""
    set from-ID from-ID + 1
  ]
  file-close
end

to save-DL-travel [ binary ]
  ifelse binary 
    [ setup-DL "DL-travel-binary.txt" ]
    [ setup-DL "DL-travel-value.txt" ]
  let from-ID 1
  while [ from-ID <= total-habitats ]                                         ; Loop through all the habitats (outside loop)
  [ file-type "  "
    let to-ID 1
    while [ to-ID <= total-habitats ]                                         ;    Loop through all the habitats  (inside loop )
    [ let success get-travel-success from-ID to-ID
      ifelse binary 
        [ ifelse success > 0 [ file-type 1 ] [ file-type 0 ] ]
        [ file-type success ]

      file-type " "
      set to-ID to-ID + 1
    ]
    file-print ""
    set from-ID from-ID + 1
  ]
  file-close
end

to save-DL-distance
  setup-DL "DL-distance.txt"
  let from-ID 1
  while [ from-ID <= total-habitats ]                                         ; Loop through all the habitats (outside loop)
  [ file-type "  "
    let to-ID 1
    while [ to-ID <= total-habitats ]                                         ;    Loop through all the habitats  (inside loop )
    [ let start-patch item ( to-ID - 1 ) [link-patches] of one-of ( habitats with [ ID = from-ID ] )
      let end-patch item ( from-ID - 1 ) [link-patches] of one-of ( habitats with [ ID = to-ID ] )

      ifelse ( from-ID != to-ID )
      [ ask start-patch  [ file-type " " file-type distance end-patch ]  ]
      [ file-type " 0" ]
      
      set to-ID to-ID + 1
    ]
    file-print ""
    set from-ID from-ID + 1
  ]
  file-close
end

to save-DL-pct [ what ]
  let file-name ( word "DL-pct-" what ".txt" )
  setup-DL file-name
  let from-ID 1
  while [ from-ID <= total-habitats ]                                         ; Loop through all the habitats (outside loop)
  [ file-type "  "
    let to-ID 1
    while [ to-ID <= total-habitats ]                                         ;    Loop through all the habitats  (inside loop )
    [ let start-patch item ( to-ID - 1 ) [link-patches] of one-of ( habitats with [ ID = from-ID ] )
      let end-patch item ( from-ID - 1 ) [link-patches] of one-of ( habitats with [ ID = to-ID ] )

      ifelse ( from-ID != to-ID )
      [ ask start-patch 
        [  file-type " " 
          calculate-inter-patch-percentages end-patch
          if ( what = "good" ) [ file-type total-good ]
          if ( what = "bad" ) [ file-type total-bad ]
          if ( what = "OK" ) [ file-type total-bad ]
        ]
      ]
      [ file-type " 0" ]

      set to-ID to-ID + 1
    ]
    file-print ""
    set from-ID from-ID + 1
  ]
  file-close
end

to calculate-inter-patch-percentages [ end-patch ]
  sprout-reporters 1                                                   ; hatch a tmp turtle to find out what is along the path
  [ set total-good 0  set total-OK 0  set total-bad 0
    set heading towards end-patch
    let my-step (distance end-patch) / 100
    let steps 0
    while [ steps < 100 ]
    [ ifelse quality = -1
      [ set total-bad total-bad + 1 ]
      [ ifelse quality = 0 
        [ set total-OK total-OK + 1 ]
        [ set total-good total-good + 1 ]
      ]
      set steps steps + 1
      fd my-step
    ]
  ]
  ask reporters [ die ]
end

to show-all-travel-paths
  cd
  let from-ID 1
  let maxval 0
  while [ from-ID <= total-habitats ]
  [ let to-ID 1 
    while [ to-ID <= total-habitats ]
    [ if ( from-ID != to-ID )
      [ let success get-travel-success from-ID to-ID
        if success > maxval [ set maxval success ]
      ]
      set to-ID to-ID + 1
    ]
    set from-ID from-ID + 1
  ]

  set from-ID 1
  while [ from-ID <= total-habitats ]
  [  show-travel-paths from-ID maxval 
    set from-ID from-ID + 1
  ]
  ask whatzits  [ draw-territory 10 ]
end

to show-travel-paths [ from-ID maxval]
  if maxval > 0
  [ create-reporters 1 
    [ set color black
      let to-ID 1                                                                         ;
      while [ to-ID <= total-habitats ]                                                   ;    Loop through all the habitats  (inside loop )
      [ if ( from-ID != to-ID )
        [ let start-patch item ( to-ID - 1 ) [link-patches] of one-of ( habitats with [ ID = from-ID ] )
          let end-patch item ( from-ID - 1 ) [link-patches] of one-of ( habitats with [ ID = to-ID ] )
          let success get-travel-success from-ID to-ID
          if ( success > 0 )
          [ pen-up
            setxy [pxcor] of start-patch [pycor] of start-patch
  ;          set color 9 - 9 * success / maxval
            set pen-size 9 * success / maxval
            ifelse ( pen-size = 0 ) [ set color 8 ] [ set color 0 ]
            pen-down
            set heading towards end-patch
            fd distance end-patch
            type from-ID type "->" type to-ID type " " print success
          ]
        ]
        set to-ID to-ID + 1
      ]
    ]
    ask reporters [ die ]
  ]
end

to save-habitat-report
  file-close
  let file-name "report-habitat.txt"
  print "Habitat report saved to: 'report-habitat.txt'"
  if file-exists? file-name [ file-delete file-name ]
  file-open file-name
;  file-print "Hab ID,Dispersers,Pop,% of Total Pop,Num Patches,% of Total Hab,Av Shortest Paths,% Edge Patches"
  file-print "Hab ID,Pop,% of Total Pop,Num Patches,% of Total Hab,Av Shortest Paths,% Edge Patches"
  let from-ID 1
  while [ from-ID <= total-habitats ]                                          ; Loop through all the habitats
  [ let the-habitat one-of habitats with [ ID = from-ID ]
    ; Print habitat ID
    file-type from-ID
    ; Print number of successful dispersers
 ;   file-type ","
 ;   file-type sum [migration-success] of the-habitat
    ; Print population in habitat
    file-type ","
    file-type count whatzits with  [ habitat-ID = the-habitat ]
    ; Print % of total population
    file-type ","
    file-type ( count whatzits with  [ habitat-ID = the-habitat ] ) / ( count whatzits ) 
    ; Print habitat area
    file-type ","
    file-type ( count patches with  [ habitat-ID = the-habitat  ] )
    ; Print proportion of total good habitat
    file-type ","
    file-type ( count patches with  [ habitat-ID = the-habitat ] ) / ( count patches with  [ quality = 1 ] ) 
    ; Print average of shortest path to other habitats
    create-reporters 1 
    [ let to-ID 1
      let total-dist 0                                                                       ;
      while [ to-ID <= total-habitats ]                                                  ;    Loop through all the habitats  (inside loop )
      [ if ( from-ID != to-ID )
        [
          let start-patch item ( to-ID - 1 ) [link-patches] of one-of ( habitats with [ ID = from-ID ] )
          let end-patch item ( from-ID - 1 ) [link-patches] of one-of ( habitats with [ ID = to-ID ] )
          setxy [pxcor] of start-patch [pycor] of start-patch
          set total-dist total-dist + distance end-patch
        ]
        set to-ID to-ID + 1
      ]
      file-type ","
      file-type total-dist / ( total-habitats )
    ]
    ; Print percent of habitat patches that are edge
    file-type ","
    file-type ( count patches with [ habitat-ID = the-habitat AND count neighbors4 with [ quality = 1 ] < 4 ] ) / ( count patches with [ habitat-ID = the-habitat] )
    file-print ""
    set from-ID from-ID + 1
  ]
  file-close
  ask reporters [ die ]
end
;to show-lineage [ line ]
;  if is-number? line [ stop ]
;  type item 0 line type ","
;  ifelse is-number? (item 1 line)
;  [ type (item 1 line) ]
;  [ type item 0 ( item 1 line ) ]
;  type ","
;  ifelse is-number? (item 2 line)
;  [ type (item 2 line) ]
;  [ type item 0 ( item 2 line ) ]
;  print ""
;  break-out item 1 line
;  break-out item 2 line
;end

to alter-habitat
  if step = At_This_Timestep
  [ ; convert string to list of habitats:  identify commas, 
    let items-in-string length Elim_These_Grn
    let greens-to-eliminate []
    let current-item 0
    while [ current-item < items-in-string ] 
    [ set greens-to-eliminate lput item current-item Elim_These_Grn greens-to-eliminate
      set current-item current-item + 1
    ]
    
    let habitats-to-convert length greens-to-eliminate
    while [ habitats-to-convert > 0 ]
    [ let habitat-to-convert item habitats-to-convert greens-to-eliminate
      ask patches with [ habitat-ID = habitat-to-convert ] 
      [ set quality 0
        set pcolor yellow
      ]
      set habitats-to-convert habitats-to-convert + 1
    ] 
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
310
10
804
525
60
60
4.0
1
10
1
1
1
0
1
1
1
-60
60
-60
60
0
0
1
ticks
30.0

BUTTON
7
241
131
274
NIL
initialize-patches
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
7
65
127
98
seed-good
seed-good
0
100
35
1
1
%
HORIZONTAL

SLIDER
7
101
127
134
seed-bad
seed-bad
0
100
35
1
1
%
HORIZONTAL

SLIDER
7
29
127
62
seed-patches
seed-patches
0
1000
114
1
1
NIL
HORIZONTAL

MONITOR
343
527
405
572
step 
step
0
1
11

SLIDER
7
137
137
170
territory-radius
territory-radius
1
10
3
1
1
cells
HORIZONTAL

SLIDER
144
29
299
62
initial-whatzits
initial-whatzits
1
500
62
1
1
NIL
HORIZONTAL

SLIDER
144
65
299
98
min-habitat-size
min-habitat-size
1
100
5
1
1
cells
HORIZONTAL

BUTTON
144
241
268
274
NIL
initialize-whatzits
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
8
718
154
751
birth-rate
birth-rate
0
10
3
1
1
per female
HORIZONTAL

BUTTON
240
286
303
319
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
240
327
303
360
go-once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
7
369
156
402
annual-survival
annual-survival
0
1
1
0.01
1
prob
HORIZONTAL

PLOT
809
10
1072
130
Population
Years
Count
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"males" 1.0 0 -13345367 true "" ""
"females" 1.0 0 -2674135 true "" ""

PLOT
809
131
1071
251
Average Age
Years
Age
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"males" 1.0 0 -13345367 true "" ""
"females" 1.0 0 -2674135 true "" ""

PLOT
809
252
1069
372
Success in Dispersing
Years
Count
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"males" 1.0 0 -13345367 true "" ""
"females" 1.0 0 -2674135 true "" ""
"die dispersing" 1.0 0 -16777216 true "" ""

SWITCH
144
101
299
134
allow-M/F-terr-overlap
allow-M/F-terr-overlap
1
1
-1000

TEXTBOX
8
10
95
28
Patch setup
11
0.0
0

SLIDER
5
639
153
672
mating-radius
mating-radius
1
20
20
1
1
cells
HORIZONTAL

MONITOR
414
528
514
573
NIL
percent-good
3
1
11

MONITOR
642
527
732
572
NIL
percent-bad
3
1
11

PLOT
809
374
1069
494
Mated
Years
Mated
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"males" 1.0 0 -13345367 true "" ""
"females" 1.0 0 -2674135 true "" ""

TEXTBOX
145
10
218
28
Whatzit Setup
11
0.0
0

MONITOR
522
528
633
573
percent-neutral
100 - percent-good - percent-bad
3
1
11

TEXTBOX
7
349
163
367
Model  Operation
11
0.0
0

BUTTON
556
634
801
667
NIL
generate-report\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

CHOOSER
557
589
801
634
choose-report
choose-report
"Lineage" "DL Travel (binary)" "DL Travel (value)" "DL Migration (binary)" "DL Migration (value)" "DL Gene Flow (binary)" "DL Gene Flow (value)" "DL distance" "DL % bad habitat between sites" "DL % good habitat between sites" "All Reports"
10

SLIDER
162
671
304
704
lineage-depth
lineage-depth
2
20
5
1
1
levels
HORIZONTAL

PLOT
810
496
1070
631
Inbreeding
Year
Wright's F
0.0
10.0
0.0
0.1
true
true
"" ""
PENS
"F" 1.0 0 -16777216 true "" ""

INPUTBOX
7
176
85
236
patch-seed
10
1
0
Number

INPUTBOX
144
177
223
237
whatzit-seed
123
1
0
Number

SLIDER
5
528
161
561
first-repro-age
first-repro-age
0
5
1
1
1
steps
HORIZONTAL

SLIDER
5
564
160
597
superior-repro-age
superior-repro-age
first-repro-age
10
4
1
1
steps
HORIZONTAL

SLIDER
131
408
266
441
death-rate-grn
death-rate-grn
0
death-rate-yel
0
0.1
1
%
HORIZONTAL

SLIDER
131
442
266
475
death-rate-yel
death-rate-yel
death-rate-grn
death-rate-red
0.1
0.1
1
%
HORIZONTAL

SLIDER
131
477
266
510
death-rate-red
death-rate-red
death-rate-yel
10
5
0.1
1
%
HORIZONTAL

SLIDER
5
600
161
633
last-repro-age
last-repro-age
superior-repro-age
100
8
1
1
steps
HORIZONTAL

CHOOSER
165
574
299
619
males-disperse
males-disperse
"Never" "1st_repro_age_only" "1st_repro_&_if_no_mates_here"
2

CHOOSER
165
622
299
667
females-disperse
females-disperse
"Never" "1st_repro_age_only" "1st_repro_&_if_no_mates_here"
1

SLIDER
162
707
345
740
Avg_Lethal_Equivalents
Avg_Lethal_Equivalents
0
20
2.2
.1
1
ea.
HORIZONTAL

TEXTBOX
141
143
303
174
* Do not change territory-radius after patches are initialized
11
0.0
1

TEXTBOX
164
364
270
409
For non-inbred individuals >1 yr old in habitable territory
11
0.0
1

TEXTBOX
18
425
125
498
During dispersal, odds of dying when traveling one cell-length through each terrain type:
11
0.0
1

TEXTBOX
8
676
163
724
* If M/F-terr-overlap not allowed, mating-radius must be > territory-radius
11
0.0
1

INPUTBOX
1156
44
1206
469
ruler
null
1
0
String

CHOOSER
165
527
308
572
Dispersal
Dispersal
"Random" "Other (see code)"
0

INPUTBOX
7
278
127
338
Elim_These_Grn
0
1
0
String (reporter)

INPUTBOX
129
278
234
338
At_This_Timestep
6
1
0
Number

MONITOR
1076
10
1149
55
Repro Fem
Repro-Fem
17
1
11

MONITOR
1077
59
1149
104
New Births
Born_this_step
17
1
11

@#$#@#$#@
## WHAT IS IT?

This is an experimental model to study the effects of different individual development parameters on future population characteristics.  The individuals are generic critters called "Whatzits".

## HOW TO RUN

1: Set patch initialization parameters and hit "initialize-patches"  
2: Set whatzit initialization parameters and hit "initialize-whatzits"  
Note that you may get stuck in an "infinite loop" if the whatzits cannot all find a suitable home.  If you are in this state, reduce the "territory-size"  
3: Set the remaining whatzit parameters  
4: Run the model with "go" or "go-once".  The latter completes a single year.

## PATCH INITIALIZATION

Whatzits live in a simple world consisting of bad (red), neutral (yellow) and good (green) habitat.  Patchs are created by seeding an empty world with a user-specified number of "seed-patches" that are randomly established in these categories based on "seed-good" and "seed-bad" percentages.  These are then grown into the surrounding areas until all cells are assigned to one of the categories.  Like-category-cells are assigned a common patch value and a total of the combined area is calculated.  A fundamental question addressed by the model involves the number and success of dispersing whatzits among the good-quality patches.

## WHATZIT INITIALIZATION

The user selects the initial number of whatzits and the territory radius of whatzits and then clicks on the "initialize-whatzits" button.  Whatzits are randomly placed in the world and assigned a random age between 0 and 10.  They are then asked to disperse according to the the dispersal rules described below (without the possibility of death).  For your visualization, Whatzits "draw" a circle around themselves to indicate their territory.

## WHATZIT PARAMETERS

"lethal equivalent" is expressed as twice the sum of the expected number of deaths ascribable to the genetic load.  Enter "Avg_Lethal_Equivalents".

## WHATZIT BEHAVIOR

Each time-step is a "year" and the simulation proceeds as follows:

1: Individuals update migration-success array  
If they have remained in the same habitat for two years, they have successfully "migrated" from their previous habitat.

2: Whatzits calculate their mating attractiveness (based on age and random numbers)  
A random number is set between 0.7 and 0.9.  The attractiveness is this number times the whatzit's age (if under 8 years old).  Times 7 if 8 years old, times 4 if 9, times 2 if 10 and times 0 if older than 10.

3: Whatzits mate  
Females older than the minimum mating age mate with the most attractive male within the territory-radius.  This male and this female calculate their Wright's inbreeding coefficient (tracking back the number of generations equal to the "lineage-depth").

4: Males that have not mated and all yearlings disperse  
Dispersal follows this process -  
1- Whatzits are assigned an energy level: age + 5  
2- Males that have not yet mated and yearlings are set as "unsettled"  
3- All unsettled whatzits are asked to:  
If male and female territories can overlap  
---- If female, become settled if there are no older females within the territory dist  
---- If male, become settled if there are no females and no older males within the territory dist  
---- If female, unsettle younger females  
---- If male, unsettle younger males  
If male and female territories can not overlap  
---- Become settled if there are no older individuals within the territory dist  
---- Unsettle younger individuals  
Increment the totall nmber of steps   
Decrease the energy by a random number betwee 0 and 1   
If the energy decreases below 0 or the total number of steps > 100, die  
4- Repeat 3 while there are still unsettled individuals

5: Unmated females attempt to mate as above

6: Females produce young  
Pregnant females give birth at the rate of the "birth-rate" variable  
Geneflow success is incremented based on the birthplace of the parents and the birthlings

7: Survival is calculated  
Whatzit's -  
- age is increased by 1  
- territory-quality is calculated as:
 ( 1 + mean of the quality of patches within territory-radius ) / 2  
- survival probability is calculated as:
 ( territory-quality * annual-survival)  
- If the survival probablility is less than a random number between 0 and 1, the whatzit dies
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
